function [LL, params, Rt, veclRt, info]=EstimateTVCopulaCL(data,distribution,corrspec,optimizer,method)
%%%------ Estimate dynamic Copula using Composite Likelihood ------------
% This function estimates the copula parameters of a time varying normal
% copula with the composite likelihood method introduced for the DCC-GARCH
% in the paper of Engle et al: "Fitting vast dimensional time - varying
% covariance models". The method is significantly faster for the normal
% copula but for the T Copula, the estimation time is not significantly
% changed.
% INPUTS
% data:          TxN matrix of U(0,1) or iid margins
% distribution:  String with values 'normal' or 'T','normal' is the default
% corrspec:      String with values 'DCC' or 'TVC', 'DCC' is the default.
%                This command describes the time varying correlation
%                equation. 'DCC' is the DCC(1,1) model of Engle - Sheppard
%                while 'TVC' is the 'TVC' model of Che - Chui.
% optimizer:     String with values 'fmincon' or 'fminunc'. The
%                unconstrained problem produces a reasonable approximation
%                of the grad and hessian, therefore if you want to
%                calculate standard errors use 'fminunc'
% method:        String with values 'CML' or 'IFM'. If data consists of
%                U(0,1) margins then there is no difference while if is iid
%                margins, the transformation to uniform is being made by
%                the empirical CDF function in 'CML' or by normcdf/tcdf in
%                case you choose IFM. Better use CML because IFM can create
%                errors.
% OUTPUTS
% LL:            The log likelihood at the optimum
% params:        The estimated parameters of the copula
% Rt:            A 3-d array of dynamic correlations
% veclRt:        The vecl of Rt
% info:          Structure that contains secondary outputs like the
%                exitflag or the grad or the hessian and the time (CPU
%                secs)
%-------------------------------------------------------------------------
% Author: Manthos Vogiatzoglou, UoM, 2008 - 2009
% contact: vogia@yahoo.com
%-------------------------------------------------------------------------
if nargin==4
    method='CML';
end
if nargin==3
    method='CML'; optimizer='fmincon';
end
if nargin==2
    method='CML'; optimizer='fmincon'; corrspec='DCC';
end
if nargin==1
    method='CML'; optimizer='fmincon'; corrspec='DCC'; distribution='normal';
end
if strcmp(distribution,'normal')==0 && strcmp(distribution,'T')==0
    error('supported distributions are normal and T')
end
if strcmp(method,'CML')==0 && strcmp(method,'IMF')==0
    error('supported methods are CML or IMF, default is CML')
end
tic
options = optimset('Display','iter','MaxFunEvals',10000,'TolCon',10^-12,'TolFun',10^-4,'TolX',10^-6);

if strcmp(distribution,'normal')==1
if strcmp(optimizer,'fmincon')
theta0=[.03;.96];
lower =.001*ones(2,1);  
upper = ones(2,1);
A=ones(1,2);
b=0.9999;
[params,likhood,exitflag]= fmincon('CompositeLLNCopula',theta0,A,b,[],[],lower,upper,[],options,data,corrspec,optimizer,method);
[compLL, Rt, veclRt] = CompositeLikDCC(params,data,corrspec, optimizer,method);
LL=-compLL;
info.time=toc;
info.exitflag=exitflag;
elseif strcmp(optimizer,'fminunc')
theta0=[1;10];
[params,likhood,exitflag,out,grad,hessian]= fminunc('CompositeLLTCopula',theta0,options,data,corrspec,optimizer,method);
[compLL, Rt, veclRt] = CompositeLikDCC(params,data,corrspec, optimizer,method);
info.grad=grad;
info.hes=hessian;
info.exitflag=exitflag;
info.ARCHparam=params(1)/(1+params(1)+params(2));
info.GARCHparam=params(2)/(1+params(1)+params(2));
LL=-compLL;
toc;
info.time=toc;
end
elseif strcmp(distribution,'T')==1
if strcmp(optimizer,'fmincon')
theta0=[12;.03;.96];
lower =[2.01;.001*ones(2,1)];
upper =[300; ones(2,1)];
A=[0 ones(1,2)];
b=0.9999;

[params,likhood,exitflag]= fmincon('CompositeLLTCopula',theta0,A,b,[],[],lower,upper,[],options,data,corrspec,optimizer,method);
[compLL, Rt, veclRt] = CompositeLLTCopula(params,data,corrspec, optimizer,method);
LL=-compLL;
info.exitflag=exitflag;
toc;
info.time=toc;
elseif strcmp(optimizer,'fminunc')
theta0=[2;1;10];
[params,likhood,exitflag,out,grad,hessian]= fminunc('CompositeLLTCopula',theta0,options,data,corrspec,optimizer,method);
[compLL, Rt, veclRt] = CompositeLLTCopula(params,data,corrspec, optimizer,method);
info.grad=grad;
info.hes=hessian;
info.exitflag=exitflag;
info.ARCHparam=params(2)/(1+params(2)+params(3));
info.GARCHparam=params(3)/(1+params(2)+params(3));
info.dof=params(1);
LL=-compLL;
toc;
info.time=toc;
end
end
fprintf(1,'elapsed CPU time in minutes = %f\n',toc/60);